/*
 * Decompiled with CFR 0.152.
 */
package com.sleepycat.je.log;

import com.sleepycat.je.log.ChecksumException;
import com.sleepycat.je.log.LogEntryType;
import com.sleepycat.je.log.LogUtils;
import com.sleepycat.je.log.Provisional;
import com.sleepycat.je.log.ReplicationContext;
import com.sleepycat.je.log.entry.LogEntry;
import com.sleepycat.je.utilint.Adler32;
import com.sleepycat.je.utilint.DbLsn;
import com.sleepycat.je.utilint.VLSN;
import java.nio.ByteBuffer;
import java.util.zip.Checksum;

public class LogEntryHeader {
    public static final int MIN_HEADER_SIZE = 14;
    public static final int MAX_HEADER_SIZE = 22;
    public static final int CHECKSUM_BYTES = 4;
    static final int ENTRYTYPE_OFFSET = 4;
    static final int FLAGS_OFFSET = 5;
    private static final int PREV_OFFSET = 6;
    private static final int ITEMSIZE_OFFSET = 10;
    public static final int VLSN_OFFSET = 14;
    private static final byte PROVISIONAL_ALWAYS_MASK = -128;
    private static final byte PROVISIONAL_BEFORE_CKPT_END_MASK = 64;
    private static final byte REPLICATED_MASK = 32;
    private static final byte INVISIBLE = 16;
    private static final byte IGNORE_INVISIBLE = -17;
    private static final byte VLSN_PRESENT = 8;
    private static final byte VERSION_6_FLAGS = -32;
    private static final byte IGNORE_VERSION_6_FLAGS = 31;
    private static final byte FILE_HEADER_TYPE_NUM = LogEntryType.LOG_FILE_HEADER.getTypeNum();
    private long checksumVal;
    private final byte entryType;
    private long prevOffset;
    private final int itemSize;
    private VLSN vlsn;
    private int entryVersion;
    private Provisional provisional;
    private boolean replicated;
    private boolean invisible;
    private boolean vlsnPresent;

    public LogEntryHeader(ByteBuffer entryBuffer, int logVersion, long lsn) throws ChecksumException {
        assert (logVersion == -1 || logVersion >= 1 && logVersion <= 15) : logVersion;
        this.checksumVal = LogUtils.readUnsignedInt(entryBuffer);
        this.entryType = entryBuffer.get();
        if (!LogEntryType.isValidType(this.entryType)) {
            throw new ChecksumException("Invalid log entry type: " + this.entryType + " lsn=" + DbLsn.getNoFormatString(lsn) + " bufPosition=" + entryBuffer.position() + " bufRemaining=" + entryBuffer.remaining());
        }
        if (this.entryType == FILE_HEADER_TYPE_NUM) {
            this.entryVersion = -1;
            entryBuffer.get();
            this.initFlags(0);
        } else {
            if (logVersion == -1) {
                throw new ChecksumException("Wrong entry type for header: " + this.entryType + " lsn=" + DbLsn.getNoFormatString(lsn) + " bufPosition=" + entryBuffer.position() + " bufRemaining=" + entryBuffer.remaining());
            }
            if (logVersion <= 6) {
                this.entryVersion = entryBuffer.get();
                this.initFlags(this.entryVersion & 0xFFFFFFE0);
                this.entryVersion &= 0x1F;
                assert (logVersion != 6 || this.entryVersion == 6);
            } else {
                this.entryVersion = logVersion;
                this.initFlags(entryBuffer.get());
            }
        }
        this.prevOffset = LogUtils.readUnsignedInt(entryBuffer);
        this.itemSize = LogUtils.readInt(entryBuffer);
        if (this.itemSize < 0) {
            throw new ChecksumException("Invalid log entry size: " + this.itemSize + " lsn=" + DbLsn.getNoFormatString(lsn) + " bufPosition=" + entryBuffer.position() + " bufRemaining=" + entryBuffer.remaining());
        }
    }

    public LogEntryHeader(LogEntry entry, Provisional provisional, ReplicationContext repContext) {
        LogEntryType logEntryType = entry.getLogType();
        this.entryType = logEntryType.getTypeNum();
        this.entryVersion = 15;
        this.itemSize = entry.getSize();
        this.provisional = provisional;
        assert (logEntryType.isReplicationPossible() || !repContext.inReplicationStream()) : logEntryType + " should never be replicated.";
        this.replicated = logEntryType.isReplicationPossible() ? repContext.inReplicationStream() : false;
        this.invisible = false;
        this.vlsnPresent = repContext.getClientVLSN() != null || repContext.mustGenerateVLSN();
    }

    public LogEntryHeader(byte entryType, int entryVersion, int itemSize, VLSN vlsn) {
        assert (vlsn != null && !vlsn.isNull()) : "vlsn = " + vlsn;
        this.entryType = entryType;
        this.entryVersion = entryVersion;
        this.itemSize = itemSize;
        this.vlsn = vlsn;
        this.replicated = true;
        this.vlsnPresent = true;
        this.provisional = Provisional.NO;
    }

    private void initFlags(int entryFlags) {
        this.provisional = (entryFlags & 0xFFFFFF80) != 0 ? Provisional.YES : ((entryFlags & 0x40) != 0 ? Provisional.BEFORE_CKPT_END : Provisional.NO);
        this.replicated = (entryFlags & 0x20) != 0;
        this.invisible = (entryFlags & 0x10) != 0;
        this.vlsnPresent = (entryFlags & 8) != 0 || this.replicated;
    }

    public void setFileHeaderVersion(int logVersion) {
        this.entryVersion = logVersion;
    }

    public long getChecksum() {
        return this.checksumVal;
    }

    public byte getType() {
        return this.entryType;
    }

    public int getVersion() {
        return this.entryVersion;
    }

    public long getPrevOffset() {
        return this.prevOffset;
    }

    public int getItemSize() {
        return this.itemSize;
    }

    public int getEntrySize() {
        return this.getSize() + this.getItemSize();
    }

    public VLSN getVLSN() {
        return this.vlsn;
    }

    public boolean getReplicated() {
        return this.replicated;
    }

    public Provisional getProvisional() {
        return this.provisional;
    }

    public boolean isInvisible() {
        return this.invisible;
    }

    public int getVariablePortionSize() {
        return 8;
    }

    public int getSize() {
        if (this.vlsnPresent) {
            return 22;
        }
        return 14;
    }

    int getSizeMinusChecksum() {
        return this.getSize() - 4;
    }

    int getInvariantSizeMinusChecksum() {
        return 10;
    }

    public void readVariablePortion(ByteBuffer entryBuffer) {
        if (this.vlsnPresent) {
            this.vlsn = new VLSN();
            this.vlsn.readFromLog(entryBuffer, this.entryVersion);
        }
    }

    public void writeToLog(ByteBuffer entryBuffer) {
        entryBuffer.position(4);
        entryBuffer.put(this.entryType);
        byte flags = 0;
        if (this.provisional == Provisional.YES) {
            flags = (byte)(flags | 0xFFFFFF80);
        } else if (this.provisional == Provisional.BEFORE_CKPT_END) {
            flags = (byte)(flags | 0x40);
        }
        if (this.replicated) {
            flags = (byte)(flags | 0x20);
        }
        if (this.vlsnPresent) {
            flags = (byte)(flags | 8);
        }
        entryBuffer.put(flags);
        entryBuffer.position(10);
        LogUtils.writeInt(entryBuffer, this.itemSize);
        if (this.vlsnPresent) {
            entryBuffer.position(entryBuffer.position() + 8);
        }
    }

    public ByteBuffer addPostMarshallingInfo(ByteBuffer entryBuffer, long lastOffset, VLSN vlsn) {
        this.prevOffset = lastOffset;
        entryBuffer.position(6);
        LogUtils.writeUnsignedInt(entryBuffer, this.prevOffset);
        if (vlsn != null) {
            this.vlsn = vlsn;
            entryBuffer.position(14);
            vlsn.writeToLog(entryBuffer);
        }
        Checksum checksum = Adler32.makeChecksum();
        checksum.update(entryBuffer.array(), entryBuffer.arrayOffset() + 4, entryBuffer.limit() - 4);
        entryBuffer.position(0);
        this.checksumVal = checksum.getValue();
        LogUtils.writeUnsignedInt(entryBuffer, this.checksumVal);
        entryBuffer.position(0);
        return entryBuffer;
    }

    public void dumpLog(StringBuilder sb, boolean verbose) {
        sb.append("<hdr ");
        this.dumpLogNoTag(sb, verbose);
        sb.append("\"/>");
    }

    void dumpLogNoTag(StringBuilder sb, boolean verbose) {
        LogEntryType lastEntryType = LogEntryType.findType(this.entryType);
        sb.append("type=\"").append(lastEntryType.toStringNoVersion()).append("/").append(this.entryVersion);
        if (this.provisional != Provisional.NO) {
            sb.append("\" prov=\"");
            sb.append((Object)this.provisional);
        }
        if (this.vlsn != null) {
            sb.append("\" ");
            this.vlsn.dumpLog(sb, verbose);
        } else {
            sb.append("\"");
        }
        if (this.getReplicated()) {
            sb.append(" isReplicated=\"1\"");
        }
        if (this.isInvisible()) {
            sb.append(" isInvisible=\"1\"");
        }
        sb.append(" prev=\"0x").append(Long.toHexString(this.prevOffset));
        if (verbose) {
            sb.append("\" size=\"").append(this.itemSize);
            sb.append("\" cksum=\"").append(this.checksumVal);
        }
    }

    void convertCommitToAbort(ByteBuffer entryBuffer) {
        assert (this.entryType == LogEntryType.LOG_TXN_COMMIT.getTypeNum());
        int itemStart = entryBuffer.position();
        int entryTypePosition = itemStart - (this.getSize() - 4);
        entryBuffer.position(entryTypePosition);
        entryBuffer.put(LogEntryType.LOG_TXN_ABORT.getTypeNum());
        Checksum checksum = Adler32.makeChecksum();
        int checksumSize = this.itemSize + (this.getSize() - 4);
        checksum.update(entryBuffer.array(), entryTypePosition + entryBuffer.arrayOffset(), checksumSize);
        entryBuffer.position(itemStart - this.getSize());
        this.checksumVal = checksum.getValue();
        LogUtils.writeUnsignedInt(entryBuffer, this.checksumVal);
    }

    public String toString() {
        StringBuilder sb = new StringBuilder();
        this.dumpLog(sb, true);
        return sb.toString();
    }

    public void dumpRep(StringBuilder sb) {
        LogEntryType lastEntryType = LogEntryType.findType(this.entryType);
        sb.append(lastEntryType.toStringNoVersion()).append("/").append(this.entryVersion);
        if (this.vlsn != null) {
            sb.append(" vlsn=").append(this.vlsn);
        } else {
            sb.append("\"");
        }
        if (this.getReplicated()) {
            sb.append(" isReplicated=\"1\"");
        }
        if (this.isInvisible()) {
            sb.append(" isInvisible=\"1\"");
        }
    }

    public boolean logicalEqualsIgnoreVersion(LogEntryHeader other) {
        return this.getVLSN().equals(other.getVLSN()) && this.getReplicated() == other.getReplicated() && this.isInvisible() == other.isInvisible() && LogEntryType.compareTypeAndVersion(this.getVersion(), this.getType(), other.getVersion(), other.getType());
    }

    public boolean isVariableLength() {
        return this.vlsnPresent;
    }

    static byte makeInvisible(byte flags) {
        flags = (byte)(flags | 0x10);
        return flags;
    }

    public static void turnOffInvisible(ByteBuffer buffer, int logHeaderStartPosition) {
        int flagsPosition = logHeaderStartPosition + 5;
        byte flags = buffer.get(flagsPosition);
        flags = (byte)(flags & 0xFFFFFFEF);
        buffer.put(flagsPosition, flags);
    }
}

