/**********************************************************************\
 © COPYRIGHT 2018 Corporation for National Research Initiatives (CNRI);
                        All rights reserved.

        The HANDLE.NET software is made available subject to the
      Handle.Net Public License Agreement, which may be obtained at
          http://hdl.handle.net/20.1000/112 or hdl:20.1000/112
\**********************************************************************/

package net.handle.server;

import java.util.Hashtable;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.io.File;
import java.lang.InterruptedException;
import java.lang.Thread;
import java.lang.reflect.InvocationTargetException;

import net.handle.util.FileSystemReadOnlyChecker;

import com.google.gson.Gson;

@SuppressWarnings({"rawtypes", "unchecked"})
public class MonitorDaemon extends Thread {

    private Hashtable monitorData = new Hashtable();
    private volatile String monitorDataString;
    private final int sleepSeconds;
    private final long startTime;
    private final Gson gson = new Gson();
    private Object sigarWrapper;

    private final AtomicLong numRequests;
    private final AtomicLong numResolutionRequests;
    private final AtomicLong numAdminRequests;
    private final AtomicLong numTxnRequests;

    private final File baseDir;

    private AtomicInteger requestsPastMinute;
    private AtomicInteger peakRequestsPerMinute;

    private volatile boolean keepRunning = true;

    public MonitorDaemon(int sleepSeconds, long startTime, AtomicLong numRequests, AtomicLong numResolutionRequests, AtomicLong numAdminRequests, AtomicLong numTxnRequests, File baseDir) {
        super("MonitorDaemon");
        setDaemon(true);
        this.sleepSeconds = sleepSeconds;
        this.startTime = startTime;

        this.numRequests = numRequests;
        this.numResolutionRequests = numResolutionRequests;
        this.numAdminRequests = numAdminRequests;
        this.numTxnRequests = numTxnRequests;

        this.baseDir = baseDir;

        initializeSigarWrapper();
    }

    public void setRequestCounters(AtomicInteger requestsPastMinute, AtomicInteger peakRequestsPerMinute) {
        this.requestsPastMinute = requestsPastMinute;
        this.peakRequestsPerMinute = peakRequestsPerMinute;
    }

    private void initializeSigarWrapper() {
        try {
            Class.forName("org.hyperic.sigar.Sigar");
            sigarWrapper = Class.forName("net.handle.server.SigarWrapper").getConstructor().newInstance();
            if (!((Boolean) (sigarWrapper.getClass().getMethod("isReady").invoke(sigarWrapper))).booleanValue()) {
                sigarWrapper = null;
            }
        } catch (Exception e) {
            //            e.printStackTrace(System.out);
            // ignore
        }
    }

    @Override
    public void run() {

        while (keepRunning) {
            try {

                monitorData = getServerInfo();
                monitorDataString = gson.toJson(monitorData);
                for (int i = 0; i < sleepSeconds && keepRunning; i++) {
                    Thread.sleep(1000);
                }
            } catch (InterruptedException e) {
                //                System.err.println("Monitor daemon interrupted: " + e);
            }
        }
    }

    public void shutdown() {
        keepRunning = false;
    }

    public String getStatusString() {
        return monitorDataString;
    }

    private Hashtable getServerInfo() {
        Hashtable systemInfo = new Hashtable();

        systemInfo.put("version", Version.version);
        if (sigarWrapper != null) {
            try {
                systemInfo.put("loadAvg", getLoadAverageInfo());
                systemInfo.put("domainName", getFQDN());
                systemInfo.put("mem", getMemInfo());
                systemInfo.put("diskInfo", getDiskInfo());
            } catch (InvocationTargetException e) {
                systemInfo.put("sigarError", e.getCause().toString());
            } catch (Exception e) {
                systemInfo.put("sigarError", e.toString());
            }
        }
        if (baseDir != null) systemInfo.put("isReadOnly", testIfFileSystemIsReadOnly());
        systemInfo.put("startTime", startTime);
        if (numRequests != null) systemInfo.put("requests", getReqInfo());
        if (requestsPastMinute != null) systemInfo.put("requestsPerMinute", getProxyReqInfo());
        systemInfo.put("lastUpdate", System.currentTimeMillis());

        return systemInfo;
    }

    private Hashtable getReqInfo() {
        Hashtable reqTable = new Hashtable();
        reqTable.put("resolution", numResolutionRequests);
        reqTable.put("admin", numAdminRequests);
        reqTable.put("txn", numTxnRequests);
        reqTable.put("total", numRequests);
        return reqTable;
    }

    private Hashtable getProxyReqInfo() {
        Hashtable reqTable = new Hashtable();
        reqTable.put("recent", requestsPastMinute.get());
        reqTable.put("peak", peakRequestsPerMinute.get());
        return reqTable;
    }

    private boolean testIfFileSystemIsReadOnly() {
        return FileSystemReadOnlyChecker.isReadOnly(baseDir);
    }

    private Object getFQDN() throws Exception {
        return sigarWrapper.getClass().getMethod("getFQDN").invoke(sigarWrapper);
    }

    private Object getLoadAverageInfo() throws Exception {
        return sigarWrapper.getClass().getMethod("getLoadAverageInfo").invoke(sigarWrapper);
    }

    private Object getDiskInfo() throws Exception {
        return sigarWrapper.getClass().getMethod("getDiskInfo").invoke(sigarWrapper);
    }

    private Object getMemInfo() throws Exception {
        return sigarWrapper.getClass().getMethod("getMemInfo").invoke(sigarWrapper);
    }

}
