/**********************************************************************\
 © COPYRIGHT 2018 Corporation for National Research Initiatives (CNRI);
                        All rights reserved.

        The HANDLE.NET software is made available subject to the
      Handle.Net Public License Agreement, which may be obtained at
          http://hdl.handle.net/20.1000/112 or hdl:20.1000/112
\**********************************************************************/

package net.handle.apps.tools;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.InetAddress;
import java.security.PrivateKey;

import com.google.gson.Gson;
import com.google.gson.stream.JsonReader;

import net.cnri.util.SimpleCommandLine;
import net.handle.hdllib.AuthenticationInfo;
import net.handle.hdllib.CreateHandleRequest;
import net.handle.hdllib.GsonUtility;
import net.handle.hdllib.HandleException;
import net.handle.hdllib.HandleRecord;
import net.handle.hdllib.HandleResolver;
import net.handle.hdllib.PublicKeyAuthenticationInfo;
import net.handle.hdllib.Util;
import net.handle.hdllib.ValueReference;

public class LoadHandlesFromJson {

    public static void printHelp() {
        System.out.println("-f\tThe path to the input json file containing handle records.");
        System.out.println();
        System.out.println("-i\tIP address of the handle server (Default 127.0.0.1).");
        System.out.println();
        System.out.println("-p\tPort of the handle server (Default 2641).");
        System.out.println();
        System.out.println("-authid\tThe index:handle of the identity performing the operation.");
        System.out.println();
        System.out.println("-privatekey\tPath to the private key for the identity performing the operation.");
        System.out.println();
    }

    public static Gson gson = GsonUtility.getPrettyGson();
    public static HandleResolver resolver = new HandleResolver();
    public static String inputFileName = null;
    public static String ipAddress = "127.0.0.1";
    public static InetAddress addr = null;
    public static int port = 2641;
    public static String authId = null;
    public static String privateKeyFileName = null;
    public static PrivateKey authPrivateKey = null;
    public static AuthenticationInfo authInfo = null;

    public static void main(String[] args) throws Exception {
        SimpleCommandLine cl = new SimpleCommandLine("f", "i", "p", "authid", "privatekey");
        cl.parse(args);

        if (cl.hasOption("f")) {
            inputFileName = cl.getOptionArgument("f");
        } else {
            System.out.println("Missing f");
            printHelp();
            return;
        }

        if (cl.hasOption("i")) {
            ipAddress = cl.getOptionArgument("i");
        } else {
            printHelp();
        }
        addr = InetAddress.getByName(ipAddress);

        if (cl.hasOption("p")) {
            port = Integer.parseInt(cl.getOptionArgument("p"));
        } else {
            printHelp();
        }

        if (cl.hasOption("authid")) {
            authId = cl.getOptionArgument("authid");
        } else {
            System.out.println("Missing authid");
            printHelp();
            return;
        }

        if (cl.hasOption("privatekey")) {
            privateKeyFileName = cl.getOptionArgument("privatekey");
            authPrivateKey = getLocalPrivateKey(privateKeyFileName, "Enter the passphrase to decrypt the private signing key: ");
            ValueReference authValueRef = ValueReference.fromString(authId);
            authInfo = new PublicKeyAuthenticationInfo(authValueRef.handle, authValueRef.index, authPrivateKey);
        } else {
            System.out.println("Missing privatekey");
            printHelp();
            return;
        }

        File inputFile = new File(inputFileName);
        @SuppressWarnings("resource")
        JsonReader reader = new JsonReader(new InputStreamReader(new FileInputStream(inputFile), "UTF-8"));
        try {
            reader.beginObject();
            while (reader.hasNext()) {
                String name = reader.nextName();
                if ("homedPrefixes".equals(name)) {
                    reader.beginArray();
                    while (reader.hasNext()) {
                        //System.out.println("HOMED PREFIX: " +reader.nextString());
                    }
                    reader.endArray();
                    continue;
                }
                if ("handleRecords".equals(name)) {
                    reader.beginObject();
                    while (reader.hasNext()) {
                        String handle = reader.nextName();
                        HandleRecord handleRecord = readNextHandleRecord(reader);
                        try {
                            createHandleRecordOnService(handleRecord);
                            System.out.println("Created: " + handle);
                        } catch (HandleException e) {
                            e.printStackTrace();
                        }
                    }
                }
            }
        } finally {
            reader.close();
        }
    }

    public static void createHandleRecordOnService(HandleRecord handleRecord) throws HandleException {
        String handle = handleRecord.getHandle();
        byte[] handleBytes = Util.encodeString(handle);
        CreateHandleRequest request = new CreateHandleRequest(handleBytes, handleRecord.getValuesAsArray(), authInfo);
        resolver.sendHdlTcpRequest(request, addr, port);
    }

    public static HandleRecord readNextHandleRecord(JsonReader reader) {
        HandleRecord handleRecord = (HandleRecord) gson.fromJson(reader, HandleRecord.class);
        return handleRecord;
    }

    public static PrivateKey getLocalPrivateKey(String key, String prompt) throws Exception {
        File privateKeyFile = new File(key);
        byte encKeyBytes[] = new byte[(int) privateKeyFile.length()];
        FileInputStream in = new FileInputStream(privateKeyFile);
        try {
            int n = 0;
            int r;
            while (n < encKeyBytes.length && (r = in.read(encKeyBytes, n, encKeyBytes.length - n)) >= 0) {
                n += r;
            }
        } finally {
            in.close();
        }

        byte keyBytes[] = null;
        byte secKey[] = null;
        if (Util.requiresSecretKey(encKeyBytes)) {
            secKey = Util.getPassphrase(prompt);
        }

        keyBytes = Util.decrypt(encKeyBytes, secKey);
        for (int i = 0; secKey != null && i < secKey.length; i++) {// clear the secret key
            secKey[i] = (byte) 0;
        }

        PrivateKey privateKey = Util.getPrivateKeyFromBytes(keyBytes, 0);
        return privateKey;
    }

    public static final String responseToPrompt(String prompt, BufferedReader in) throws IOException {
        System.out.print("\n" + prompt + ": ");
        System.out.flush();
        return in.readLine().trim();
    }
}
