/**********************************************************************\
 © COPYRIGHT 2018 Corporation for National Research Initiatives (CNRI);
                        All rights reserved.

        The HANDLE.NET software is made available subject to the
      Handle.Net Public License Agreement, which may be obtained at
          http://hdl.handle.net/20.1000/112 or hdl:20.1000/112
\**********************************************************************/

package net.handle.apps.test;

import net.handle.hdllib.*;
import java.io.*;

public class ClientTest {
    private static HandleResolver resolver = new HandleResolver();

    private static final String OSNAME = System.getProperty("os.name");
    private static boolean windows = false;
    private static boolean linux = false;

    /***************************testGlobalSites******************************
    - Tests each global site
    - Checks each interface (TCP, HTTP, UDP)
    - Prints to screen if interface passed or failed
     *********************************************************************/
    static void testGlobalSites() {

        Interface serverInterface;
        resolver.setTcpTimeout(10000); // use 10 second timeouts
        // resolver.traceMessages = true;

        SiteInfo GlobalSites[] = resolver.getConfiguration().getGlobalSites();

        for (int i = 0; i < GlobalSites.length; i++) {
            for (int j = 0; j < GlobalSites[i].servers.length; j++) {
                System.out.println("\nSERVER: " + GlobalSites[i].servers[j].getAddressString());

                //test each interface (TCP, HTTP, UDP)
                for (int k = 0; k < GlobalSites[i].servers[j].interfaces.length; k++) {
                    serverInterface = GlobalSites[i].servers[j].interfaces[k];

                    try {
                        GenericRequest req = new GenericRequest(Common.BLANK_HANDLE, AbstractMessage.OC_GET_SITE_INFO, null);
                        resolver.sendRequestToInterface(req, GlobalSites[i].servers[j], serverInterface);
                        System.out.print(Interface.protocolName(serverInterface.protocol));
                        System.out.println(":" + serverInterface.port + "                   ***PASS***");
                    } catch (HandleException t) {
                        System.err.print(Interface.protocolName(serverInterface.protocol));
                        System.err.print(":" + serverInterface.port + "                   ***FAIL***");
                        System.err.println("  " + HandleException.getCodeStr(t.getCode()));
                    }
                }
                ping(GlobalSites[i].servers[j].getAddressString());
            }
        }
    }

    /***************************testNASites******************************
    - Tests the site corresponding to the given prefix handle
    - Checks each interface (TCP, HTTP, UDP)
    - Prints to screen if interface passed or failed
     *********************************************************************/
    static void testNASites(String naHandle) {
        Interface serverInterface;
        SiteInfo localSites[];
        try {
            GenericRequest greq = new GenericRequest(Util.encodeString(naHandle), AbstractMessage.OC_GET_SITE_INFO, null);
            localSites = resolver.findLocalSites(greq);
        } catch (HandleException t) {
            System.err.println("Error resolving prefix handle");
            return;
        }

        for (int i = 0; i < localSites.length; i++) {
            for (int j = 0; j < localSites[i].servers.length; j++) {
                System.out.println("SERVER: " + localSites[i].servers[j].getAddressString());

                //test each interface (TCP, HTTP, UDP)
                for (int k = 0; k < localSites[i].servers[j].interfaces.length; k++) {
                    serverInterface = localSites[i].servers[j].interfaces[k];

                    try {
                        GenericRequest req = new GenericRequest(Common.BLANK_HANDLE, AbstractMessage.OC_GET_SITE_INFO, null);
                        resolver.sendRequestToInterface(req, localSites[i].servers[j], serverInterface);

                        System.out.print(Interface.protocolName(serverInterface.protocol));
                        System.out.println(":" + serverInterface.port + "                   ***PASS***");
                    } catch (HandleException t) {
                        System.err.print(Interface.protocolName(serverInterface.protocol));
                        System.err.print(":" + serverInterface.port + "                   ***FAIL***");
                        System.err.println("  " + HandleException.getCodeStr(t.getCode()));
                    }
                }
                ping(localSites[i].servers[j].getAddressString());
            }
        }
    }

    /***********************************************************************
    PING
    - pings a server
    - can ping using windows, linux, or solaris os's
    - the os defaults to solaris
     ************************************************************************/
    static void ping(String server) {

        String osName = OSNAME.toUpperCase().trim();

        //possible locations of ping
        String windowsPaths[] = { "C:\\WINNT\\SYSTEM32\\", "C:\\WINNT\\", "C:\\WINDOWS\\SYSTEM\\", "C:\\WINDOWS\\", "c:\\windows\\system32\\ping.exe" };
        String linuxPaths[] = { "/bin/", "/usr/bin/", "/sbin/", "/usr/sbin/" };

        //used for os names like "windows nt" or "windows 2000"
        int spcIdx = osName.indexOf(' ');
        if (spcIdx != -1) osName = osName.substring(0, spcIdx);

        //if os name is not linux, windows, or solaris, the default is solaris
        if (osName.length() > 7 || osName.length() < 5) {
            linux = false;
            windows = false;
        } else if ((osName.substring(0, 5)).equals("LINUX")) {
            linux = true;
            windows = false;
        } else if (osName.length() == 7 && (osName.substring(0, 7)).equals("WINDOWS")) {
            windows = true;
            linux = false;
        } else {
            linux = false;
            windows = false;
        }

        String pingPath = null;

        if (windows) { //locate ping in windows
            for (int i = 0; i < windowsPaths.length; i++) {
                try {
                    File pingDir = new File(windowsPaths[i], "ping.exe");
                    if (pingDir.exists()) {
                        pingPath = windowsPaths[i];
                        break;
                    }
                } catch (Exception e) {
                    if (i == windowsPaths.length - 1) {
                        System.out.println("UNABLE TO LOCATE PING");
                        break;
                    }
                }
            }
        } else { //locate ping in linux or solaris
            for (int i = 0; i < linuxPaths.length; i++) {
                try {
                    File pingDir = new File(linuxPaths[i], "ping");
                    if (pingDir.exists()) {
                        pingPath = linuxPaths[i];
                        break;
                    }
                } catch (Exception e) {
                    if (i == linuxPaths.length - 1) {
                        System.out.println("UNABLE TO LOCATE PING");
                        break;
                    }
                }
            }
        }
        Runtime runtime = Runtime.getRuntime();

        if (pingPath != null) {
            if (windows) { //ping server in windows
                String cmd[] = { pingPath.concat("ping"), "-n", "10", server };
                try {
                    Process p = runtime.exec(cmd);
                    BufferedReader reader = new BufferedReader(new InputStreamReader(new DataInputStream(new BufferedInputStream(p.getInputStream()))));

                    String lineRead = null;
                    int avgIdx = -1;
                    int lossIdx = -1;
                    int lossIdx2 = -1;
                    String avg = "error";
                    String loss = "error";

                    //parse ping summary
                    //only print average time and packet loss
                    while ((lineRead = reader.readLine()) != null) {
                        avgIdx = lineRead.indexOf("Average");
                        lossIdx = lineRead.indexOf("%");
                        lossIdx2 = lineRead.indexOf("(");
                        if (avgIdx != -1) {
                            avgIdx = lineRead.indexOf("=", avgIdx);
                            avg = lineRead.substring(avgIdx + 3);
                        }
                        if (lossIdx != -1 && lossIdx2 != -1) loss = lineRead.substring(lossIdx2 + 1, lossIdx);
                    }
                    //if packet loss is >60%, fail
                    if (Integer.parseInt(loss) > 60) {
                        System.out.print("PING [avg:" + avg);
                        System.out.println("/loss:" + loss + "%]          ***FAIL***");
                    }
                    //if unable to parse ping summary correctly, assume pass
                    else if (avg.equals("error") || loss.equals("error")) System.out.println("PING                ***PASS***");
                    else {
                        System.out.print("PING [avg:" + avg);
                        System.out.println("/loss:" + loss + "%]      ***PASS***");
                    }
                    //if exception, the arguments to ping are probably wrong
                } catch (Exception e) {
                    System.err.println("Error pinging " + server + "    ");
                }
            } else { //ping server in linux or solaris
                String cmd[] = null;
                String cmdL[] = { pingPath.concat("ping"), "-c", "10", server };
                String cmdS[] = { pingPath.concat("ping"), "-s", server, "56", "10" };
                if (linux) cmd = cmdL;
                else cmd = cmdS;
                try {
                    Process p = runtime.exec(cmd);
                    BufferedReader reader = new BufferedReader(new InputStreamReader(new DataInputStream(new BufferedInputStream(p.getInputStream()))));

                    String lineRead = null;
                    int avgIdx = -1;
                    int lossIdx = -1;
                    int lossIdx2 = -1;
                    String avg = null;
                    String loss = null;

                    //parse ping summary
                    //only print average time and packet loss
                    while ((lineRead = reader.readLine()) != null) {
                        avgIdx = lineRead.indexOf(" = ");
                        lossIdx2 = lineRead.indexOf("received");
                        lossIdx = lineRead.indexOf("%");
                        if (avgIdx != -1) {
                            avgIdx = lineRead.indexOf("/", avgIdx);
                            avg = lineRead.substring(avgIdx + 1, lineRead.indexOf("/", avgIdx + 1));
                        }
                        if (lossIdx != -1 && lossIdx2 != -1) {
                            loss = lineRead.substring(lossIdx2 + 10, lossIdx);
                        }
                    }
                    //if packet loss is >30%, fail
                    if (loss != null && Integer.parseInt(loss) > 30) {
                        System.out.print("PING [avg:" + avg);
                        System.out.println("/loss:" + loss + "%]          ***FAIL***");
                    }
                    //if unable to parse ping summary correctly
                    else if (avg == null && loss == null) System.out.println("PING                  ***N/A***");
                    else {
                        System.out.print("PING [avg:" + avg + "ms");
                        System.out.println("/loss:" + loss + "%]      ***PASS***");
                    }
                    //if exception, the arguments to ping are probably wrong
                } catch (Exception e) {
                    e.printStackTrace();
                    System.err.println("Error pinging " + server + ":    " + e);
                }
            }
        } else //ping was not in one of the specified directories
            System.out.println("UNABLE TO LOCATE PING");
    }
}
